# StyleSeat - Installation & Setup Guide

This guide will walk you through setting up the StyleSeat application on your local machine or server.

## Prerequisites

Before starting, ensure you have:
- **PHP 8.2+** installed and configured
- **Composer** (dependency manager for PHP)
- **Node.js 16+** and **npm** (for frontend assets)
- **MySQL 5.7+** database server
- **Git** (for version control)
- A **Firebase project** (for authentication)
- A **Stripe account** (for payments)

### Check Versions
```bash
php -v              # Should show PHP 8.2+
composer -v         # Should show Composer version
node -v            # Should show Node 16+
npm -v             # Should show npm version
mysql --version    # Should show MySQL 5.7+
```

## Step 1: Clone/Download the Project

```bash
cd ~/path/to/projects
# If using Git:
git clone <repository-url> styleseatApp
cd styleseatApp
```

Or extract the provided ZIP file to your projects folder.

## Step 2: Install PHP Dependencies

```bash
composer install
```

This installs all required PHP packages including Laravel, Firebase Admin SDK, Stripe Cashier, and others.

## Step 3: Install JavaScript Dependencies

```bash
npm install
```

This installs Tailwind CSS, Alpine.js, and Vite for frontend asset building.

## Step 4: Environment Configuration

### Copy the example environment file:
```bash
cp .env.example .env
```

### Generate application key:
```bash
php artisan key:generate
```

### Edit .env file with your configuration:
```env
APP_NAME=StyleSeat
APP_ENV=local
APP_DEBUG=true
APP_URL=http://localhost:8000

# Database
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=styleseat_db
DB_USERNAME=root
DB_PASSWORD=

# Firebase
FIREBASE_PROJECT_ID=your-firebase-project-id
FIREBASE_API_KEY=your-firebase-api-key
FIREBASE_AUTH_DOMAIN=your-firebase-project.firebaseapp.com
FIREBASE_DATABASE_URL=https://your-firebase-project.firebasedatabase.app
FIREBASE_STORAGE_BUCKET=your-firebase-project.appspot.com
FIREBASE_MESSAGING_SENDER_ID=your-messaging-sender-id
FIREBASE_APP_ID=your-firebase-app-id
FIREBASE_MEASUREMENT_ID=your-measurement-id
FIREBASE_CREDENTIALS_PATH=./firebase-credentials.json

# Stripe
STRIPE_PUBLIC_KEY=pk_test_your_test_public_key
STRIPE_SECRET_KEY=sk_test_your_test_secret_key
STRIPE_WEBHOOK_SECRET=whsec_test_your_webhook_secret

# Mail
MAIL_MAILER=log
MAIL_HOST=mailpit
MAIL_PORT=1025
MAIL_USERNAME=null
MAIL_PASSWORD=null
MAIL_ENCRYPTION=null
MAIL_FROM_ADDRESS="hello@example.com"

# App Settings
ADMIN_EMAIL=admin@styleseat.local
PLATFORM_COMMISSION_PERCENTAGE=20
```

## Step 5: Setup Firebase

### Create Firebase Project:
1. Go to [Firebase Console](https://firebase.google.com/console)
2. Create a new project named "StyleSeat"
3. Enable **Authentication** (Email/Password & Google)
4. Create **Firestore Database** (optional for real-time features)

### Download Service Account Key:
1. Go to Project Settings → Service Accounts
2. Click "Generate New Private Key"
3. Save the JSON file as `firebase-credentials.json` in the project root
4. Update `FIREBASE_CREDENTIALS_PATH` in `.env` to point to this file

### Get Firebase Config:
1. Go to Project Settings → General
2. Copy your Firebase config values
3. Update `FIREBASE_*` variables in `.env`

## Step 6: Setup Stripe

### Get API Keys:
1. Go to [Stripe Dashboard](https://dashboard.stripe.com)
2. Navigate to Developers → API Keys
3. Copy **Publishable Key** and **Secret Key** (Test Mode)
4. Update `STRIPE_PUBLIC_KEY` and `STRIPE_SECRET_KEY` in `.env`

### Get Webhook Secret (optional for development):
1. In Developers section, go to Webhooks
2. Add endpoint: `http://localhost:8000/api/stripe/webhook`
3. Copy the signing secret
4. Update `STRIPE_WEBHOOK_SECRET` in `.env`

## Step 7: Create Database

### Create MySQL Database:
```bash
mysql -u root
> CREATE DATABASE styleseat_db;
> EXIT;
```

Or using a GUI tool like phpMyAdmin.

## Step 8: Run Migrations & Seeders

### Run migrations to create tables:
```bash
php artisan migrate
```

### Seed database with mock data:
```bash
php artisan db:seed
```

This creates:
- 1 Admin account (admin@styleseat.local)
- 75 Client users
- 25 Professional users
- 250+ Bookings
- 1000+ Transactions
- All necessary configuration data

## Step 9: Create Storage Symlink

```bash
php artisan storage:link
```

This creates a symbolic link from `storage/app/public` to `public/storage` for file access.

## Step 10: Build Frontend Assets

### Development build with hot reload:
```bash
npm run dev
```

In another terminal, keep this running while developing.

### Production build:
```bash
npm run build
```

## Step 11: Start Development Server

In a new terminal window:
```bash
php artisan serve
```

The application will be available at `http://localhost:8000`

## Step 12: Test the Application

### Access the application:
- **Home**: http://localhost:8000
- **Search**: http://localhost:8000/search
- **Admin Dashboard**: http://localhost:8000/admin/dashboard

### Demo Credentials:

**Admin Account:**
- Email: `admin@styleseat.local`
- Role: Admin
- (No password needed in development mode if using Firebase auth)

**Sample Professional:**
- Any user with email ending in professional domain
- Access: `/professional/dashboard`

**Sample Client:**
- Any user with email
- Access: `/client/dashboard`

### Test Stripe Payments:
Use these test card numbers:
- **Visa**: `4242 4242 4242 4242`
- **MasterCard**: `5555 5555 5555 4444`
- **Decline**: `4000 0000 0000 0002`
- **Expiry**: Any future date
- **CVC**: Any 3 digits

## Development Workflow

### Start developing:
```bash
# Terminal 1: Watch and compile frontend
npm run dev

# Terminal 2: Run Laravel development server
php artisan serve
```

### Common commands:
```bash
# Clear cache
php artisan cache:clear

# Refresh database with fresh data
php artisan migrate:refresh --seed

# Create new migration
php artisan make:migration create_new_table

# Create new model
php artisan make:model ModelName -m

# Create new controller
php artisan make:controller ControllerName --resource

# Tinker (interactive shell)
php artisan tinker
```

## Production Deployment

### cPanel Deployment:

1. **Prepare files:**
   ```bash
   composer install --no-dev --optimize-autoloader
   npm run build
   php artisan config:cache
   php artisan route:cache
   ```

2. **Upload files** via SFTP (exclude node_modules, .git)

3. **Create .env** on server with production values:
   ```env
   APP_ENV=production
   APP_DEBUG=false
   STRIPE_MODE=live
   ```

4. **Set permissions:**
   ```bash
   chmod -R 755 storage bootstrap/cache
   ```

5. **Run migrations:**
   ```bash
   php artisan migrate --force
   ```

6. **Configure web server** to point document root to `public/`

## Troubleshooting

### Database Connection Error:
```bash
# Check database credentials in .env
# Ensure MySQL is running
mysql -u root -p
```

### Firebase Token Verification Failed:
- Ensure `firebase-credentials.json` exists in project root
- Verify Firebase project ID matches in `.env`
- Check that Firebase Admin SDK is properly installed

### Stripe Payment Not Working:
- Verify API keys in `.env` are from Test mode
- Check that Stripe keys are not swapped
- Ensure Stripe PHP SDK is installed via Composer

### Assets Not Loading:
```bash
npm run build
php artisan storage:link
```

### Composer Issues:
```bash
composer clear-cache
composer install
```

### Permission Denied Errors:
```bash
chmod -R 755 storage bootstrap/cache
chmod 755 public
```

## Support

For detailed information, see:
- [README.md](README.md) - Project overview
- [PROJECT_SUMMARY.md](PROJECT_SUMMARY.md) - Feature summary

## Quick Start Checklist

- [ ] PHP 8.2+ installed
- [ ] Composer installed
- [ ] Node.js 16+ installed
- [ ] MySQL running
- [ ] Project cloned/extracted
- [ ] `composer install` completed
- [ ] `npm install` completed
- [ ] `.env` file configured
- [ ] Firebase project created and configured
- [ ] Stripe account created and keys added
- [ ] Database created: `styleseat_db`
- [ ] `php artisan migrate` completed
- [ ] `php artisan db:seed` completed
- [ ] `php artisan storage:link` completed
- [ ] `npm run build` completed
- [ ] `php artisan serve` started
- [ ] Application accessible at `http://localhost:8000`

---

**Ready to develop!** The application is now ready for local development. Start with `npm run dev` and `php artisan serve`.
