# StyleSeat Live Deployment Guide

## Overview

This guide covers deploying StyleSeat to a live web server (shared hosting or VPS) for production testing. The application is designed for cPanel/LAMP environments, but can also run on dedicated servers.

## Prerequisites

Before deploying, ensure you have:

1. **Server Access**
   - Web hosting account (shared hosting with cPanel OR VPS)
   - SSH/Terminal access
   - FTP or SFTP access (if no SSH)
   - PHP 8.2+ installed
   - MySQL database

2. **Domain & SSL**
   - Domain name (or subdomain)
   - SSL certificate (auto-generated via Let's Encrypt is fine)
   - DNS records pointing to your server

3. **Third-Party Services**
   - Firebase project configured
   - Stripe account (test mode for initial testing)
   - Email service (Mailgun, SMTP provider, or localhost)

4. **Local Preparation**
   - Application built and tested locally
   - All assets built (npm run build)
   - .env configured with production settings
   - Database migrations verified locally

## Deployment Steps

### Step 1: Prepare Your Local Application

#### 1.1 Build Assets for Production
```bash
# In your project directory
npm run build
```
This compiles Tailwind CSS, JavaScript, and other assets into `public/` directory.

#### 1.2 Create .env for Production
Copy `.env.example` to `.env.production` and configure:

```env
APP_NAME=StyleSeat
APP_ENV=production
APP_DEBUG=false
APP_KEY=base64:xxxxx  # From key:generate

APP_URL=https://yourdomain.com

# Database
DB_CONNECTION=mysql
DB_HOST=your-server-host
DB_PORT=3306
DB_DATABASE=your-db-name
DB_USERNAME=your-db-user
DB_PASSWORD=your-db-password

# Firebase (from Firebase Console)
FIREBASE_PROJECT_ID=your-project-id
FIREBASE_PRIVATE_KEY_ID=your-key-id
FIREBASE_PRIVATE_KEY="-----BEGIN PRIVATE KEY-----\n...\n-----END PRIVATE KEY-----\n"
FIREBASE_DATABASE_URL=https://your-project.firebaseio.com
FIREBASE_AUTH_DOMAIN=your-project.firebaseapp.com
FIREBASE_STORAGE_BUCKET=your-project.appspot.com
FIREBASE_MESSAGING_SENDER_ID=your-sender-id
FIREBASE_APP_ID=your-app-id

# Stripe
STRIPE_PUBLIC_KEY=pk_test_xxxxx
STRIPE_SECRET_KEY=sk_test_xxxxx

# Mail Configuration
MAIL_MAILER=smtp
MAIL_HOST=smtp.mailtrap.io  # or your email service
MAIL_PORT=465
MAIL_USERNAME=your-username
MAIL_PASSWORD=your-password
MAIL_FROM_ADDRESS=noreply@yourdomain.com
MAIL_FROM_NAME="StyleSeat"

# Session
SESSION_DRIVER=cookie
CACHE_DRIVER=file
QUEUE_CONNECTION=sync

# Optional: For advanced logging
LOG_CHANNEL=stack
LOG_LEVEL=warning
```

#### 1.3 Optimize Application
```bash
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

### Step 2: Upload to Server

#### 2.1 Using SFTP (Recommended)
1. **Connect to Server**
   - Host: your-domain.com
   - Username: your-cpanel-username or ssh-user
   - Password: your-password
   - Port: 22 (SSH) or 21 (SFTP)

2. **Upload Files**
   ```
   Upload entire project EXCEPT:
   - .env (do NOT upload - create on server)
   - node_modules/ (not needed on server)
   - .git/ (optional - exclude for security)
   - .venv/ (Python virtual env - not needed)
   - vendor/ (we'll install with Composer)
   
   Create this structure on server:
   public_html/      (or your web root)
   └── app/
   └── bootstrap/
   └── config/
   └── database/
   └── resources/
   └── routes/
   └── storage/
   └── tests/
   └── artisan
   └── composer.json
   └── composer.lock
   └── tailwind.config.js
   └── vite.config.js
   ... and all other files except those listed above
   ```

#### 2.2 Using Git (If Server Has Git)
```bash
# On server, in your web root
git clone https://github.com/your-repo/styleseat.git .
# or
git clone your-repo-url && cd styleseat
```

#### 2.3 Using Archive (ZIP)
1. Create ZIP locally (excluding node_modules, .env, vendor)
2. Upload ZIP to server
3. Extract on server via cPanel or SSH:
   ```bash
   unzip styleseat.zip
   ```

### Step 3: Configure on Server

#### 3.1 Create .env on Server
Via cPanel File Manager or SSH:
```bash
cd /home/username/public_html
nano .env
```

Paste your production .env configuration. Save and exit (Ctrl+X, Y, Enter in nano).

#### 3.2 Set Directory Permissions
```bash
chmod 755 /home/username/public_html
chmod 755 /home/username/public_html/public
chmod 755 /home/username/public_html/storage
chmod 755 /home/username/public_html/bootstrap/cache

# Make writable
chmod 777 /home/username/public_html/storage
chmod 777 /home/username/public_html/bootstrap/cache
chmod 644 /home/username/public_html/.env
```

#### 3.3 Install Dependencies
Via SSH (recommended):
```bash
cd /home/username/public_html
php composer.phar install --no-dev --optimize-autoloader
# or
composer install --no-dev --optimize-autoloader
```

**Note**: If using shared hosting without SSH, use cPanel's Terminal or create a deployment script.

#### 3.4 Create Database Tables
```bash
php artisan migrate --force
```

**Optional**: Seed test data
```bash
php artisan db:seed --force
```

#### 3.5 Clear Caches
```bash
php artisan cache:clear
php artisan config:clear
php artisan view:clear
php artisan route:clear
```

### Step 4: Configure Web Server

#### 4.1 For cPanel/Apache
Create `.htaccess` in `public/` directory (usually already exists):

```apache
<IfModule mod_rewrite.c>
    <IfModule mod_negotiation.c>
        Options -MultiViews -Indexes
    </IfModule>

    RewriteEngine On

    # Handle Authorization Header
    RewriteCond %{HTTP:Authorization} .
    RewriteRule .* - [E=HTTP_AUTHORIZATION:%{HTTP:Authorization}]

    # Redirect Trailing Slashes If Not A Folder...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_URI} (.+)/$
    RewriteRule ^ %1 [L,R=301]

    # Send Requests To Front Controller...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteRule ^ index.php [L]
</IfModule>
```

#### 4.2 For Nginx
Add to server block:
```nginx
location / {
    try_files $uri $uri/ /index.php?$query_string;
}

location ~ \.php$ {
    fastcgi_pass unix:/var/run/php-fpm.sock;
    fastcgi_index index.php;
    fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
    include fastcgi_params;
}
```

#### 4.3 Verify Web Root
Ensure your web root points to `public/` directory, NOT the project root.

**cPanel**:
- Go to Addon Domains or Public HTML
- Set document root to `/home/username/public_html/public`

### Step 5: Configure SSL/HTTPS

#### 5.1 In cPanel
1. Go to "AutoSSL"
2. Click "Check for "YYYY" status"
3. Wait for certificate generation (5-10 minutes)

#### 5.2 Redirect HTTP to HTTPS
Add to `.env`:
```env
APP_URL=https://yourdomain.com
```

Add to `app/Http/Middleware/TrustProxies.php`:
```php
protected $proxiedHeaders = Request::HEADER_X_FORWARDED_FOR |
    Request::HEADER_X_FORWARDED_HOST |
    Request::HEADER_X_FORWARDED_PROTO |
    Request::HEADER_X_FORWARDED_AWS_ELB;
```

Add to `routes/web.php`:
```php
if (env('APP_ENV') === 'production') {
    URL::forceScheme('https');
}
```

### Step 6: Test Installation

#### 6.1 Run Web Installer
Visit your domain in browser:
```
https://yourdomain.com/installer
```

This will guide you through the setup wizard:
1. ✅ Check system requirements
2. ✅ Configure database (should auto-fill from .env)
3. ✅ Configure app settings
4. ✅ Run migrations
5. ✅ Create admin user
6. ✅ Finalize installation

#### 6.2 Access Application
After installer completes:
```
https://yourdomain.com
```

#### 6.3 Log In
- Use admin email/password created in installer
- Access admin dashboard: `/admin/dashboard`

### Step 7: Post-Deployment Configuration

#### 7.1 Set Up Email
Test email sending:
```bash
php artisan tinker
> Mail::raw('Test email', function($message) { 
    $message->to('your@email.com'); 
  });
```

#### 7.2 Configure Firebase
1. Go to Firebase Console
2. Project Settings → Service Accounts
3. Copy credentials into .env:
   ```env
   FIREBASE_PROJECT_ID=...
   FIREBASE_PRIVATE_KEY=...
   FIREBASE_DATABASE_URL=...
   ```

#### 7.3 Configure Stripe
1. Log into Stripe Dashboard
2. Get API keys (test mode for initial testing)
3. Update .env:
   ```env
   STRIPE_PUBLIC_KEY=pk_test_...
   STRIPE_SECRET_KEY=sk_test_...
   ```

#### 7.4 Set Up Cron Jobs
For scheduled tasks (email, cleanup, etc.):

**Via cPanel**:
1. Go to Cron Jobs
2. Add entry:
```bash
* * * * * cd /home/username/public_html && php artisan schedule:run >> /dev/null 2>&1
```

This runs every minute and executes scheduled tasks defined in `app/Console/Kernel.php`.

#### 7.5 Configure File Storage
For user uploads (profile pics, portfolio images):

**Option 1**: Local Storage (Default)
```env
FILESYSTEM_DISK=local
```
Files stored in `storage/app/public/`

**Option 2**: AWS S3
```env
FILESYSTEM_DISK=s3
AWS_ACCESS_KEY_ID=...
AWS_SECRET_ACCESS_KEY=...
AWS_DEFAULT_REGION=us-east-1
AWS_BUCKET=your-bucket
```

Make storage accessible:
```bash
php artisan storage:link
```

### Step 8: Performance Optimization

#### 8.1 Enable Caching
```bash
php artisan cache:clear
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

Update `config/cache.php`:
```php
'default' => env('CACHE_DRIVER', 'file'),
```

#### 8.2 Database Optimization
```bash
# On server, via SSH
mysql -u your-user -p your-database < optimize.sql
```

Or run Laravel's model optimizer:
```bash
php artisan model:cache
```

#### 8.3 Enable Gzip Compression
Add to `.htaccess`:
```apache
<IfModule mod_deflate.c>
  AddOutputFilterByType DEFLATE text/html text/plain text/xml
  AddOutputFilterByType DEFLATE text/css text/javascript
  AddOutputFilterByType DEFLATE application/javascript
</IfModule>
```

#### 8.4 Configure Headers
Add to `app/Http/Middleware/SetCacheHeaders.php`:
```php
public function handle(Request $request, Closure $next)
{
    $response = $next($request);
    
    // Cache static assets for 1 year
    if ($request->is('css/*', 'js/*', 'images/*')) {
        $response->header('Cache-Control', 'public, max-age=31536000, immutable');
    } else {
        $response->header('Cache-Control', 'public, max-age=3600, must-revalidate');
    }
    
    return $response;
}
```

### Step 9: Monitoring & Maintenance

#### 9.1 Monitor Logs
```bash
tail -f storage/logs/laravel.log
```

Or via cPanel: File Manager → storage/logs/

#### 9.2 Check Status
```bash
php artisan optimize
php artisan down --secret="12345"  # Maintenance mode
php artisan up                     # Back online
```

#### 9.3 Backup Database
Via cPanel:
1. Go to Backups
2. Download backup
3. Or automate via SSH:
```bash
mysqldump -u user -p database > backup-$(date +%Y%m%d).sql
```

#### 9.4 Update Application
```bash
git pull origin main
composer install --no-dev
php artisan migrate --force
php artisan cache:clear
```

## Common Issues & Solutions

### Issue 1: 500 Internal Server Error
**Solution**:
```bash
php artisan config:cache
php artisan cache:clear
```
Check logs: `tail -f storage/logs/laravel.log`

### Issue 2: Database Connection Failed
**Verify in .env**:
```bash
DB_HOST=localhost      # or server IP
DB_DATABASE=correct_db
DB_USERNAME=correct_user
DB_PASSWORD=correct_pass
```

Test connection:
```bash
php artisan db
```

### Issue 3: Installer Not Accessible
**Ensure CheckInstallation middleware is active**:
```bash
php artisan migrate --force
```

Delete flag and restart:
```bash
rm storage/installer_completed.flag
```

### Issue 4: Static Assets (CSS/JS) Not Loading
**Ensure public directory is web root**:
```bash
ln -s /home/username/public_html/public /home/username/public_html/public
php artisan storage:link
```

### Issue 5: Email Not Sending
**Test configuration**:
```bash
php artisan tinker
> Mail::raw('Test', fn($m) => $m->to('test@email.com'));
```

Check logs for SMTP errors.

### Issue 6: Stripe Payments Not Working
**Verify keys in .env**:
```env
STRIPE_PUBLIC_KEY=pk_test_...
STRIPE_SECRET_KEY=sk_test_...
```

Test webhook: Stripe Dashboard → Webhooks → Send test event

### Issue 7: Firebase Authentication Failing
**Verify credentials in .env** and Firebase console:
```bash
php artisan tinker
> app('firebase.client')->getAuth()->createUser(['email' => 'test@test.com', 'password' => 'test123']);
```

## Deployment Checklist

- [ ] Local build completed (`npm run build`)
- [ ] `.env` configured for production (APP_KEY already set)
- [ ] All files uploaded to server
- [ ] Permissions set correctly (755/777)
- [ ] Composer installed (`composer install --no-dev`)
- [ ] .env created on server
- [ ] Database migrations run (`php artisan migrate --force`)
- [ ] Web installer accessed and completed
- [ ] Admin user created
- [ ] SSL/HTTPS configured
- [ ] Domain pointing to correct web root
- [ ] Email configured and tested
- [ ] Firebase credentials configured
- [ ] Stripe keys configured (test mode)
- [ ] Cron jobs set up
- [ ] Storage linked (`php artisan storage:link`)
- [ ] Logs checked for errors
- [ ] Performance optimizations applied
- [ ] Backup system configured
- [ ] Admin dashboard accessible
- [ ] Test booking created end-to-end

## Next Steps

After deployment:
1. **Test Core Features**: Create test bookings, make test payments (Stripe test mode)
2. **Load Testing**: Use tools like Apache Bench or JMeter to test capacity
3. **Security Audit**: Run OWASP ZAP or similar security scanner
4. **Performance Testing**: Check Google PageSpeed Insights, GTmetrix
5. **User Testing**: Have actual users test the system
6. **Switch to Production**: Move Stripe from test mode to live when ready

## Support

For issues:
1. Check `storage/logs/laravel.log` for detailed error messages
2. Review system requirements via installer: `https://yourdomain.com/installer/requirements`
3. Test database connection via installer: `https://yourdomain.com/installer/database`
4. Enable debug mode temporarily (.env: `APP_DEBUG=true`) for more detailed errors

---

**StyleSeat v1.0 - Live Deployment**  
*Ready for production testing on shared hosting or VPS*
