<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use Illuminate\View\View;
use Illuminate\Http\Request;

class TransactionController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:admin');
    }

    public function index(): View
    {
        $transactions = Transaction::with('client.user', 'professional.user', 'booking')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.transactions.index', ['transactions' => $transactions]);
    }

    public function show($transactionId): View
    {
        $transaction = Transaction::with('client.user', 'professional.user', 'booking')
            ->findOrFail($transactionId);

        return view('admin.transactions.show', ['transaction' => $transaction]);
    }

    public function filter(Request $request): View
    {
        $query = Transaction::with('client.user', 'professional.user', 'booking');

        if ($request->filled('status')) {
            $query->where('status', $request->get('status'));
        }

        if ($request->filled('from_date')) {
            $query->whereDate('created_at', '>=', $request->get('from_date'));
        }

        if ($request->filled('to_date')) {
            $query->whereDate('created_at', '<=', $request->get('to_date'));
        }

        $transactions = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('admin.transactions.index', ['transactions' => $transactions]);
    }

    public function exportCsv()
    {
        $transactions = Transaction::with('client.user', 'professional.user', 'booking')->get();

        $headers = [
            'Content-Type' => 'text/csv; charset=utf-8',
            'Content-Disposition' => 'attachment; filename=transactions.csv',
        ];

        $callback = function () use ($transactions) {
            $file = fopen('php://output', 'w');
            
            fputcsv($file, [
                'ID', 'Stripe Payment ID', 'Client', 'Professional',
                'Amount', 'Platform Fee', 'Professional Payout', 'Currency',
                'Status', 'Created At'
            ]);

            foreach ($transactions as $transaction) {
                fputcsv($file, [
                    $transaction->id,
                    $transaction->stripe_payment_id,
                    $transaction->client->user->name,
                    $transaction->professional->user->name,
                    $transaction->amount,
                    $transaction->platform_fee,
                    $transaction->professional_payout,
                    $transaction->currency,
                    $transaction->status,
                    $transaction->created_at,
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
