<?php

namespace App\Http\Controllers;

use App\Models\ProfessionalProfile;
use App\Models\Booking;
use App\Models\Review;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\JsonResponse;

class ProfessionalController extends Controller
{
    public function show($id): View
    {
        $professional = ProfessionalProfile::with(['user', 'services', 'reviews', 'gallery'])
            ->where('is_published', true)
            ->findOrFail($id);

        $upcomingBookings = $professional->bookings()
            ->where('scheduled_at', '>', now())
            ->where('status', '!=', 'cancelled')
            ->count();

        $isFavorite = auth()->check() && 
            $professional->favorites()->where('client_id', auth()->id())->exists();

        return view('professional.profile', [
            'professional' => $professional,
            'upcomingBookings' => $upcomingBookings,
            'isFavorite' => $isFavorite,
        ]);
    }

    public function getAvailability($id, Request $request): JsonResponse
    {
        $professional = ProfessionalProfile::findOrFail($id);
        $date = $request->get('date');

        // Get availability for the requested date
        $dayOfWeek = \Carbon\Carbon::parse($date)->dayOfWeek;
        $availability = $professional->availability()
            ->where('day_of_week', $dayOfWeek)
            ->first();

        if (!$availability) {
            return response()->json(['times' => []]);
        }

        // Generate time slots (30 min intervals)
        $startTime = \Carbon\Carbon::parse($availability->start_time);
        $endTime = \Carbon\Carbon::parse($availability->end_time);
        $times = [];

        while ($startTime < $endTime) {
            $times[] = $startTime->format('H:i');
            $startTime->addMinutes(30);
        }

        return response()->json(['times' => $times]);
    }

    public function toggleFavorite($id): JsonResponse
    {
        $professional = ProfessionalProfile::findOrFail($id);
        
        if (auth()->user()->favorites()->where('professional_id', $id)->exists()) {
            auth()->user()->favorites()->detach($id);
            $isFavorite = false;
        } else {
            auth()->user()->favorites()->attach($id);
            $isFavorite = true;
        }

        return response()->json(['isFavorite' => $isFavorite]);
    }
}
