<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ProfessionalProfile extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id', 'business_name', 'bio', 'city', 'address', 'phone',
        'stripe_account_id', 'stripe_payout_method', 'rating',
        'total_bookings', 'total_revenue', 'is_verified', 'is_published',
        'specialties', 'policies',
    ];

    protected $casts = [
        'specialties' => 'array',
        'policies' => 'array',
        'is_verified' => 'boolean',
        'is_published' => 'boolean',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function services()
    {
        return $this->hasMany(Service::class);
    }

    public function bookings()
    {
        return $this->hasMany(Booking::class);
    }

    public function availability()
    {
        return $this->hasMany(Availability::class);
    }

    public function blockedTimes()
    {
        return $this->hasMany(BlockedTime::class);
    }

    public function gallery()
    {
        return $this->hasMany(Gallery::class)->orderBy('sort_order');
    }

    public function reviews()
    {
        return $this->hasMany(Review::class);
    }

    public function favorites()
    {
        return $this->belongsToMany(User::class, 'favorites', 'professional_id', 'client_id');
    }

    public function transactions()
    {
        return $this->hasMany(Transaction::class);
    }

    public function updateRating()
    {
        $avgRating = $this->reviews()->avg('rating') ?? 5.0;
        $this->update(['rating' => round($avgRating, 2)]);
    }

    public function getAverageRating()
    {
        return round($this->reviews()->avg('rating') ?? 5.0, 1);
    }

    public function getReviewCount()
    {
        return $this->reviews()->count();
    }

    public function getTotalRevenue()
    {
        return $this->transactions()
            ->where('status', 'succeeded')
            ->sum('professional_payout');
    }

    public function getTotalBookings()
    {
        return $this->bookings()->where('status', '!=', 'cancelled')->count();
    }
}
