<?php

namespace App\Services;

use App\Models\Booking;
use App\Models\Transaction;
use App\Models\ProfessionalProfile;

class BookingService
{
    protected $stripeService;

    public function __construct(StripeService $stripeService)
    {
        $this->stripeService = $stripeService;
    }

    public function createBooking($clientId, $professionalId, $serviceId, $scheduledAt, $notes = null)
    {
        $service = \App\Models\Service::findOrFail($serviceId);
        
        $booking = Booking::create([
            'client_id' => $clientId,
            'professional_id' => $professionalId,
            'service_id' => $serviceId,
            'scheduled_at' => $scheduledAt,
            'duration' => $service->duration,
            'city' => \App\Models\ProfessionalProfile::findOrFail($professionalId)->city,
            'amount' => $service->price,
            'status' => 'pending',
            'notes' => $notes,
        ]);

        return $booking;
    }

    public function processPayment(Booking $booking, $paymentIntentId)
    {
        $paymentIntent = $this->stripeService->confirmPayment($paymentIntentId);

        if ($paymentIntent->status !== 'succeeded') {
            return false;
        }

        $amount = $booking->amount;
        $platformFee = $amount * (config('app.platform_commission_percentage', 20) / 100);
        $profitAmount = $amount - $platformFee;

        Transaction::create([
            'booking_id' => $booking->id,
            'client_id' => $booking->client_id,
            'professional_id' => $booking->professional_id,
            'stripe_payment_id' => $paymentIntent->id,
            'stripe_payment_intent' => $paymentIntentId,
            'amount' => $amount,
            'platform_fee' => $platformFee,
            'professional_payout' => $profitAmount,
            'currency' => $paymentIntent->currency,
            'status' => 'succeeded',
            'raw_payload' => $paymentIntent->toArray(),
        ]);

        $booking->update(['status' => 'confirmed']);

        return true;
    }

    public function confirmBooking(Booking $booking)
    {
        $booking->confirm();
    }

    public function cancelBooking(Booking $booking, $reason = null)
    {
        $booking->cancel($reason);

        // Optionally refund if payment was made
        if ($booking->transaction && $booking->transaction->isSucceeded()) {
            $this->stripeService->refundPayment(
                $booking->transaction->stripe_payment_intent,
                $booking->amount
            );
        }
    }

    public function completeBooking(Booking $booking)
    {
        $booking->update([
            'status' => 'completed',
            'completed_at' => now(),
        ]);
    }
}
